/*
 * Copyright @ 2001-2003, The Institute for Genomic Research (TIGR).
 * All rights reserved.
 *
 *
 * ColumnWorker.java
 *
 * Created on September 27, 2001, 12:39 PM
 * @author  wliang
 * @version
 */

package org.tigr.midas.util;

import java.io.*;
import java.util.*;
import java.math.*;
import java.lang.*;
import org.tigr.midas.Constant;
import org.tigr.midas.math.MathTools;
//import org.tigr.midas.internalDB.*;
//import org.tigr.midas.exception.*;
import org.tigr.midas.util.mevFileParser.*;

public class ColumnWorker {
    
    private String fileName;
    private int count;
    private int nonZeroCount;
    private int[] metaRow;
    private int[] metaCol;
    private String[] metaCombo;
    private long colOneTotal;
    private long colTwoTotal;
    private float col12Ratio;
    private float[] colOneArrayRaw;
    private float[] colTwoArrayRaw;
    private float[] colOneArray;
    private float[] colTwoArray;
    private float[] plainRatioArray;
    private float[] logProdArray;
    private float[] logRatioArray;
    private float[] colOneBkgdArray;
    private float[] colTwoBkgdArray;
    private float[] snRatioOne;
    private float[] snRatioTwo;
    private boolean[] rowFlag;
    private boolean[] channelFlag1;
    private boolean[] channelFlag2;
    private int[] colThreeArray;
    private String[] colThreeArrayStr;
    private String[] colFourArrayStr;
    private String[] featNameArray;
    private int[] ndxArray;
    private String[] uidArrayStr;
    
    private MevFileParser mfp;
    public int file_type;
    public boolean repNotCombined = true;
    
    
    /** Mod for MeV 
     *  Specialized constructor to support MeV
     */
    public ColumnWorker(float [] column1, float [] column2, String [] MetaCombo){
        this(column1.length);
        colOneArray = column1;
        colTwoArray = column2;
        metaCombo = MetaCombo;
        count = colOneArray.length;
        
        for(int rowCount = 0; rowCount < colOneArray.length; rowCount++){
            if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1){
                colOneTotal += colOneArray[rowCount];
                colTwoTotal += colTwoArray[rowCount];
            }
            this.rowFlag[rowCount] = true;
        }
        col12Ratio = (float)colOneTotal / (float)colTwoTotal;
    }
    
    
    public ColumnWorker(String fName) throws Exception{
        if (fName.endsWith(".mev")){
            file_type = Constant.MEV_TYPE;
            fileName = fName;
            mfp = new MevFileParser();
            File thisFile = new File(fName);
            try{
                if (MevFileParser.validate(thisFile) == MevFileParser.MEV_FILE){
                    mfp.loadFile(thisFile);
                    Vector uidVec = mfp.getColumnNamed("UID");
                    initializeArrays(uidVec.size());
                    
                    //Create and populate database
                    //Schema schema = new Schema();
                    //for (int i = 0; i < uidVec.size(); i++){
                    //    schema.populateTable( (i+1), uidVec.elementAt(i).toString());
                    //}
                    //schema.query ("select * from spot");
                }else{
                    System.out.println("Invalid MEV file: " + fName);
                    //throw new InvalidMevFormatException();
                }
            }catch(Exception ex){
                ex.printStackTrace();
            }
        }else{ //.tav etc.
            file_type = Constant.TAV_TYPE;
            fileName = fName;
            int cnt = 0;
            String ln = "";
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    ln = inFile.readLine();
                    cnt++;
                }while(ln.length() != 0);
            }catch (NullPointerException e){
                //System.out.println("    Total gene count = " + (cnt-1));
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            initializeArrays(cnt-1);
        }
    }
    
    public ColumnWorker(int size){
        metaRow = new int[size];
        metaCol = new int[size];
        metaCombo = new String[size];
        colOneArray = new float[size];
        colTwoArray = new float[size];
        colOneArrayRaw = new float[size];
        colTwoArrayRaw = new float[size];
        colOneBkgdArray = new float[size];
        colTwoBkgdArray = new float[size];
        snRatioOne = new float[size];
        snRatioTwo = new float[size];
        rowFlag = new boolean[size];
        channelFlag1 = new boolean[size];
        channelFlag2 = new boolean[size];
        colThreeArray = new int[size];
        colThreeArrayStr = new String[size];
        colFourArrayStr = new String[size];
        featNameArray = new String[size];
        ndxArray = new int[size];
        uidArrayStr = new String[size];
        count = 0;
        colOneTotal = 0;
        colTwoTotal = 0;
        col12Ratio = 0;
    }
    
    public void initializeArrays(int size) {
        metaRow = new int[size];
        metaCol = new int[size];
        metaCombo = new String[size];
        colOneArray = new float[size];
        colTwoArray = new float[size];
        colOneArrayRaw = new float[size];
        colTwoArrayRaw = new float[size];
        colOneBkgdArray = new float[size];
        colTwoBkgdArray = new float[size];
        snRatioOne = new float[size];
        snRatioTwo = new float[size];
        rowFlag = new boolean[size];
        channelFlag1 = new boolean[size];
        channelFlag2 = new boolean[size];
        colThreeArray = new int[size];
        colThreeArrayStr = new String[size];
        colFourArrayStr = new String[size];
        featNameArray = new String[size];
        ndxArray = new int[size];
        uidArrayStr = new String[size];
        count = 0;
        colOneTotal = 0;
        colTwoTotal = 0;
        col12Ratio = 0;
    }
    
    //Set columns for metaRow and metaColumn
    public void setColumnsForMeta(){
        try{
            Vector v1 = mfp.getColumnNamed("MR");
            Vector v2 = mfp.getColumnNamed("MC");
            for (int rowCount = 0; rowCount < v1.size(); rowCount++){
                metaCombo[rowCount] = v1.elementAt(rowCount).toString() + v2.elementAt(rowCount).toString();
            }
            count = v1.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumnsForMeta(int col1, int col2) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForMeta();
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    metaCombo[rowCount] = inputRowVec.elementAt(col1).toString() + "_" + inputRowVec.elementAt(col2).toString();
                    rowCount ++;
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            //Re-size the arraies
            metaCombo = new ArrayResizer(metaCombo, rowCount).getSArray();
            metaRow = new ArrayResizer(metaRow, rowCount).getIArray();
            metaCol = new ArrayResizer(metaCol, rowCount).getIArray();
            
            count = rowCount;
        }
    }
    
    //Set columns for UID
    public void setColumnsForUID(){
        if (this.file_type == Constant.MEV_TYPE){
            try{
                Vector v1 = mfp.getColumnNamed("UID");
                uidArrayStr = VecToArray.toString(v1);
                count = v1.size();
            }catch(FieldNotFoundException ex){
                //System.out.println("    !! UID not found!");
                ex.printStackTrace();
            }catch(Exception ex){
                //System.out.println("    !! UIDs not copied to vector!");
                ex.printStackTrace();
            }
        }else{
            uidArrayStr = null;
        }
    }
    
    //Set columns for featNameArray
    public void setColumnsForFeatName(){
        try{
            Vector v1 = mfp.getColumnNamed("FeatN");
            featNameArray = VecToArray.toString(v1);
            count = v1.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumnForFeatName(int col) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForFeatName();
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    featNameArray[rowCount] = inputRowVec.elementAt(col).toString();
                    rowCount ++;
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            //Re-size the arraies
            featNameArray = new ArrayResizer(featNameArray, rowCount).getSArray();
            
            count = rowCount;
        }
    }
    
    //Set single float type column
    public void setOneColumn(int col) throws FileNotFoundException, NumberFormatException, Exception{
        
        Vector inputRowVec = new Vector();
        String line = "";
        int rowCount = 0;
        
        try{
            BufferedReader inFile = new BufferedReader(new FileReader(fileName));
            do{
                line = inFile.readLine();
                StringSplitter st = new StringSplitter(line, '\t');
                inputRowVec = new Vector();
                while (st.hasMoreTokens()) {
                    inputRowVec.add(st.nextToken());
                }
                colOneArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col).toString());
                rowCount ++;
            }while(line.length() != 0);
        }catch (NullPointerException e){
        }catch (FileNotFoundException nfex){
            throw nfex;
        }catch (NumberFormatException nex){
            throw nex;
        }catch (Exception ex){
            ex.printStackTrace();
            throw ex;
        }
        
        //Re-size the arraies
        colOneArray = new ArrayResizer(colOneArray, rowCount).getFArray();
        
        count = rowCount;
    }
    
    /***********************************************************************
     *  setColumns
     *
     *  Set cy3 and cy5 values to arraies, without considering flags
     *
     ***********************************************************************/
    public void setColumnsForIAIB(){
        try{
            Vector aVec = mfp.getColumnNamed("IA");
            Vector bVec = mfp.getColumnNamed("IB");
            colOneArray = VecToArray.toFloat(aVec);
            colTwoArray = VecToArray.toFloat(bVec);
            for (int rowCount = 0; rowCount < colOneArray.length; rowCount++){
                colOneArrayRaw[rowCount] = colOneArray[rowCount];
                colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                channelFlag1[rowCount] = true;
                channelFlag2[rowCount] = true;
                rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                
                if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                    colOneTotal += colOneArray[rowCount];
                    colTwoTotal += colTwoArray[rowCount];
                }
            }
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            count = aVec.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumns(int col1, int col2) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForIAIB();
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    colOneArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col1).toString());
                    colTwoArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col2).toString());
                    //System.out.println("colOneArray[" + rowCount + "] = " + colOneArray[rowCount] + ", colTwoArray[" + rowCount + "] = " + colTwoArray[rowCount]);
                    colOneArrayRaw[rowCount] = colOneArray[rowCount];
                    colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                    channelFlag1[rowCount] = true;
                    channelFlag2[rowCount] = true;
                    rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                    
                    if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                        colOneTotal += colOneArray[rowCount];
                        colTwoTotal += colTwoArray[rowCount];
                    }
                    rowCount ++;
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            //System.out.println("***** col12Ratio = " + col12Ratio + ", colOneTotal = " + colOneTotal + ", colTwoTotal = " + colTwoTotal);
            count = rowCount;
            
            //Re-size the arraies
            colOneArray = new ArrayResizer(colOneArray, rowCount).getFArray();
            colTwoArray = new ArrayResizer(colTwoArray, rowCount).getFArray();
            colOneArrayRaw = new ArrayResizer(colOneArrayRaw, rowCount).getFArray();
            colTwoArrayRaw = new ArrayResizer(colTwoArrayRaw, rowCount).getFArray();
            channelFlag1 = new ArrayResizer(channelFlag1, rowCount).getBArray();
            channelFlag2 = new ArrayResizer(channelFlag2, rowCount).getBArray();
            rowFlag = new ArrayResizer(rowFlag, rowCount).getBArray();
        }
    }
    
    /***********************************************************************
     *  setColumns
     *
     *  Set cy3 and cy5 values to arraies, using only one flag col3 as filter
     *  for both col1 and col2
     *
     ***********************************************************************/
    public void setColumnsForIAIBwithOneFlag(){
        try{
            Vector aVec = mfp.getColumnNamed("IA");
            Vector bVec = mfp.getColumnNamed("IB");
            Vector f1Vec = mfp.getColumnNamed("Flag");
            colOneArray = VecToArray.toFloat(aVec);
            colTwoArray = VecToArray.toFloat(bVec);
            colThreeArrayStr = VecToArray.toString(f1Vec);
            for (int rowCount = 0; rowCount < colOneArray.length; rowCount++){
                colOneArrayRaw[rowCount] = colOneArray[rowCount];
                colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                if ((colThreeArrayStr[rowCount].equalsIgnoreCase("B")) || (colThreeArrayStr[rowCount].equalsIgnoreCase("C"))){
                    channelFlag1[rowCount] = true;
                    channelFlag2[rowCount] = true;
                }else{
                    channelFlag1[rowCount] = false;
                    channelFlag2[rowCount] = false;
                }
                rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                if (!rowFlag[rowCount]){
                    colOneArray[rowCount] = 0;
                    colTwoArray[rowCount] = 0;
                }
                if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                    colOneTotal += colOneArray[rowCount];
                    colTwoTotal += colTwoArray[rowCount];
                }
            }
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            count = aVec.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumns(int col1, int col2, int col3) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForIAIBwithOneFlag();
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    colOneArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col1).toString());
                    colTwoArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col2).toString());
                    colOneArrayRaw[rowCount] = colOneArray[rowCount];
                    colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                    colThreeArrayStr[rowCount] = inputRowVec.elementAt(col3).toString();
                    
                    if ((colThreeArrayStr[rowCount].equalsIgnoreCase("B")) || (colThreeArrayStr[rowCount].equalsIgnoreCase("C"))){
                        channelFlag1[rowCount] = true;
                        channelFlag2[rowCount] = true;
                    }else{
                        channelFlag1[rowCount] = false;
                        channelFlag2[rowCount] = false;
                    }
                    
                    rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                    if (!rowFlag[rowCount]){
                        colOneArray[rowCount] = 0;
                        colTwoArray[rowCount] = 0;
                    }
                    
                    if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                        colOneTotal += colOneArray[rowCount];
                        colTwoTotal += colTwoArray[rowCount];
                    }
                    rowCount ++;
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            count = rowCount;
            
            //Re-size the arraies
            colOneArray = new ArrayResizer(colOneArray, rowCount).getFArray();
            colTwoArray = new ArrayResizer(colTwoArray, rowCount).getFArray();
            colOneArrayRaw = new ArrayResizer(colOneArrayRaw, rowCount).getFArray();
            colTwoArrayRaw = new ArrayResizer(colTwoArrayRaw, rowCount).getFArray();
            colThreeArrayStr = new ArrayResizer(colThreeArrayStr, rowCount).getSArray();
            channelFlag1 = new ArrayResizer(channelFlag1, rowCount).getBArray();
            channelFlag2 = new ArrayResizer(channelFlag2, rowCount).getBArray();
            rowFlag = new ArrayResizer(rowFlag, rowCount).getBArray();
        }
    }
    
    
    /***********************************************************************
     *  setColumns
     *
     *  Set cy3 and cy5 values to arraies, using col3 as filter for col1 and
     *  col4 for col2
     *
     ***********************************************************************/
    public void setColumnsForIAIBwithTwoFlags(){
        try{
            Vector aVec = mfp.getColumnNamed("IA");
            Vector bVec = mfp.getColumnNamed("IB");
            Vector f1Vec = mfp.getColumnNamed("FlagA");
            Vector f2Vec = mfp.getColumnNamed("FlagB");
            colOneArray = VecToArray.toFloat(aVec);
            colTwoArray = VecToArray.toFloat(bVec);
            colThreeArrayStr = VecToArray.toString(f1Vec);
            colFourArrayStr = VecToArray.toString(f2Vec);
            
            for (int rowCount = 0; rowCount < colOneArray.length; rowCount++){
                colOneArrayRaw[rowCount] = colOneArray[rowCount];
                colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                
                if ((colThreeArrayStr[rowCount].equalsIgnoreCase("B")) || (colThreeArrayStr[rowCount].equalsIgnoreCase("C"))){
                    channelFlag1[rowCount] = true;
                }else{
                    channelFlag1[rowCount] = false;
                }
                
                if ( ((colFourArrayStr[rowCount].equalsIgnoreCase("B")) || (colFourArrayStr[rowCount].equalsIgnoreCase("C")))){
                    channelFlag2[rowCount] = true;
                }else{
                    channelFlag2[rowCount] = false;
                }
                
                rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                if (!rowFlag[rowCount]){
                    colOneArray[rowCount] = 0;
                    colTwoArray[rowCount] = 0;
                }
                if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                    colOneTotal += colOneArray[rowCount];
                    colTwoTotal += colTwoArray[rowCount];
                }
            }
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            count = aVec.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumns(int col1, int col2, int col3, int col4) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForIAIBwithTwoFlags();
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    colOneArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col1).toString());
                    colTwoArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col2).toString());
                    colOneArrayRaw[rowCount] = colOneArray[rowCount];
                    colTwoArrayRaw[rowCount] = colTwoArray[rowCount];
                    colThreeArrayStr[rowCount] = inputRowVec.elementAt(col3).toString();
                    colFourArrayStr[rowCount] = inputRowVec.elementAt(col4).toString();
                    
                    if ( ((colThreeArrayStr[rowCount].equalsIgnoreCase("B")) || (colThreeArrayStr[rowCount].equalsIgnoreCase("C")))){
                        channelFlag1[rowCount] = true;
                    }else{
                        channelFlag1[rowCount] = false;
                    }
                    
                    if ( ((colFourArrayStr[rowCount].equalsIgnoreCase("B")) || (colFourArrayStr[rowCount].equalsIgnoreCase("C")))){
                        channelFlag2[rowCount] = true;
                    }else{
                        channelFlag2[rowCount] = false;
                    }
                    
                    rowFlag[rowCount] = channelFlag1[rowCount] && channelFlag2[rowCount];
                    if (!rowFlag[rowCount]){
                        colOneArray[rowCount] = 0;
                        colTwoArray[rowCount] = 0;
                    }
                    
                    if (colOneArray[rowCount] > 1 && colTwoArray[rowCount] > 1 && rowFlag[rowCount]){
                        colOneTotal += colOneArray[rowCount];
                        colTwoTotal += colTwoArray[rowCount];
                    }
                    
                    rowCount ++;
                    
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            col12Ratio = (float)colOneTotal / (float)colTwoTotal;
            count = rowCount;
            
            //Re-size the arraies
            colOneArray = new ArrayResizer(colOneArray, rowCount).getFArray();
            colTwoArray = new ArrayResizer(colTwoArray, rowCount).getFArray();
            colOneArrayRaw = new ArrayResizer(colOneArrayRaw, rowCount).getFArray();
            colTwoArrayRaw = new ArrayResizer(colTwoArrayRaw, rowCount).getFArray();
            colThreeArrayStr = new ArrayResizer(colThreeArrayStr, rowCount).getSArray();
            colFourArrayStr = new ArrayResizer(colFourArrayStr, rowCount).getSArray();
            channelFlag1 = new ArrayResizer(channelFlag1, rowCount).getBArray();
            channelFlag2 = new ArrayResizer(channelFlag2, rowCount).getBArray();
            rowFlag = new ArrayResizer(rowFlag, rowCount).getBArray();
        }
    }
    
    public void setColumnsForTwoBkgd(float snRatioCutoff){
        try{
            Vector aVec = mfp.getColumnNamed("IA");
            colOneArray = VecToArray.toFloat(aVec);
            Vector bVec = mfp.getColumnNamed("IB");
            colTwoArray = VecToArray.toFloat(bVec);
            Vector v1 = mfp.getColumnNamed("BkgA");
            colOneBkgdArray = VecToArray.toFloat(v1);
            Vector v2 = mfp.getColumnNamed("BkgB");
            colTwoBkgdArray = VecToArray.toFloat(v2);
            
            for (int rowCount = 0; rowCount < colOneArray.length; rowCount++){
                snRatioOne[rowCount] = (colOneArray[rowCount] + colOneBkgdArray[rowCount]) / colOneBkgdArray[rowCount];
                snRatioTwo[rowCount] = (colTwoArray[rowCount] + colTwoBkgdArray[rowCount]) / colTwoBkgdArray[rowCount];
                
                if ( rowFlag[rowCount] && (snRatioOne[rowCount] >= snRatioCutoff) && (snRatioTwo[rowCount] >= snRatioCutoff) ){
                    rowFlag[rowCount] = true;
                }else{
                    rowFlag[rowCount] = false;
                }
                
                if (!rowFlag[rowCount]){
                    colOneArray[rowCount] = 0;
                    colTwoArray[rowCount] = 0;
                }
            }
            count = v1.size();
        }catch(FieldNotFoundException ex){
            ex.printStackTrace();
        }
    }
    
    public void setColumnsForTwoBkgd(int col1, int col2, float snRatioCutoff) throws FileNotFoundException, NumberFormatException, Exception{
        if (this.file_type == Constant.MEV_TYPE){
            setColumnsForTwoBkgd(snRatioCutoff);
        }else{
            Vector inputRowVec = new Vector();
            String line = "";
            int rowCount = 0;
            
            try{
                BufferedReader inFile = new BufferedReader(new FileReader(fileName));
                do{
                    line = inFile.readLine();
                    StringSplitter st = new StringSplitter(line, '\t');
                    inputRowVec = new Vector();
                    while (st.hasMoreTokens()) {
                        inputRowVec.add(st.nextToken());
                    }
                    colOneBkgdArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col1).toString());
                    colTwoBkgdArray[rowCount] = Float.parseFloat(inputRowVec.elementAt(col2).toString());
                    
                    snRatioOne[rowCount] = (colOneArray[rowCount] + colOneBkgdArray[rowCount]) / colOneBkgdArray[rowCount];
                    snRatioTwo[rowCount] = (colTwoArray[rowCount] + colTwoBkgdArray[rowCount]) / colTwoBkgdArray[rowCount];
                    
                    if ( rowFlag[rowCount] && (snRatioOne[rowCount] >= snRatioCutoff) && (snRatioTwo[rowCount] >= snRatioCutoff) ){
                        rowFlag[rowCount] = true;
                    }else{
                        rowFlag[rowCount] = false;
                    }
                    
                    if (!rowFlag[rowCount]){
                        colOneArray[rowCount] = 0;
                        colTwoArray[rowCount] = 0;
                    }
                    
                    rowCount ++;
                }while(line.length() != 0);
            }catch (NullPointerException e){
            }catch (FileNotFoundException nfex){
                throw nfex;
            }catch (NumberFormatException nex){
                throw nex;
            }catch (Exception ex){
                ex.printStackTrace();
                throw ex;
            }
            
            //Re-size the arraies
            colOneBkgdArray = new ArrayResizer(colOneBkgdArray, rowCount).getFArray();
            colTwoBkgdArray = new ArrayResizer(colTwoBkgdArray, rowCount).getFArray();
            snRatioOne = new ArrayResizer(snRatioOne, rowCount).getFArray();
            snRatioTwo = new ArrayResizer(snRatioTwo, rowCount).getFArray();
        }
    }
    
    public void setRowCount(int cnt){
        count = cnt;
    }
    
    public void KeepTwoColumnsAddFlags(long cutOff1, long cutOff2){
        nonZeroCount = 0;
        for(int i = 0; i < count; i++) {
            if ((colTwoArray[i] <= 1) || (colOneArray[i] <= 1)){
                colOneArray[i] = 0;
                colTwoArray[i] = 0;
                channelFlag1[i]  = false;
                channelFlag2[i]  = false;
                rowFlag[i] = false;
            }
            
            if ((colOneArray[i] > cutOff1) && (colTwoArray[i] > cutOff2)){
                rowFlag[i] = true;
                nonZeroCount++;
            }else{
                rowFlag[i] = false;
            }
        }
    }
    
    public void resetTwoColumnsAddFlagsRefColOne(long cutOff1, long cutOff2){
        nonZeroCount = 0;
        for(int i = 0; i < count; i++) {
            if ((colTwoArray[i] <= 1) || (colOneArray[i] <= 1)){
                colOneArray[i] = 0;
                colTwoArray[i] = 0;
                channelFlag1[i]  = false;
                channelFlag2[i]  = false;
                rowFlag[i] = false;
            }else{
                colTwoArray[i] *= col12Ratio;
            }
            
            if ((colOneArray[i] > cutOff1) && (colTwoArray[i] > cutOff2)){
                rowFlag[i] = true;
                nonZeroCount++;
            }else{
                rowFlag[i] = false;
            }
        }
    }
    
    public void resetTwoColumnsAddFlagsRefColTwo(long cutOff1, long cutOff2){
        nonZeroCount = 0;
        for(int i = 0; i < count; i++) {
            if ((colOneArray[i] <= 1) || (colTwoArray[i] <= 1)){
                colOneArray[i] = 0;
                colTwoArray[i] = 0;
                channelFlag1[i]  = false;
                channelFlag2[i]  = false;
                rowFlag[i] = false;
            }else{
                colOneArray[i] /= col12Ratio;
            }
            
            if ((colOneArray[i] > cutOff1) && (colTwoArray[i] > cutOff2)){
                rowFlag[i] = true;
                nonZeroCount++;
            }else{
                rowFlag[i] = false;
            }
        }
    }
    
    public void computeGoodRatiosAndLogRatios(long cutOff1, long cutOff2){
        plainRatioArray = new float[colOneArray.length];
        logProdArray = new float[colOneArray.length];
        logRatioArray = new float[colOneArray.length];
        for (int i = 0; i < count; i++){
            if ((colOneArray[i] > cutOff1) && (colTwoArray[i] > cutOff2) && (rowFlag[i])){
                plainRatioArray[i] = colTwoArray[i] / colOneArray[i];
                logProdArray[i] = MathTools.log10(colOneArray[i] * colTwoArray[i]);
                logRatioArray[i] = MathTools.log2(colTwoArray[i] / colOneArray[i]);
            }else{
                plainRatioArray[i] = 0;
                logProdArray[i] = 0;
                logRatioArray[i] = 9999.f;
            }
        }
    }
    
    public String getFileName(){
        if (fileName.lastIndexOf(FileBrowser.fsep) > 0){
            return fileName.substring(fileName.lastIndexOf(FileBrowser.fsep)+1, fileName.lastIndexOf('.'));
        }else{
            return fileName;
        }
    }
    
    public String getFileNameExtension(){
        if (fileName.lastIndexOf(FileBrowser.fsep) > 0){
            return fileName.substring(fileName.lastIndexOf('.'), fileName.length());
        }else{
            return "";
        }
    }
    
    public final String[] getMetaCombo(){
        return metaCombo;
    }
    
    public final int[] getMetaRow(){
        return metaRow;
    }
    
    public final int[] getMetaCol(){
        return metaCol;
    }
    
    public final String[] getFeatNameArray(){
        return featNameArray;
    }
    
    public final float[] getColumnOneArrayRaw() {
        return colOneArrayRaw;
    }
    
    public final float[] getColumnTwoArrayRaw() {
        return colTwoArrayRaw;
    }
    
    public final float[] getColumnOneArray() {
        return colOneArray;
    }
    
    public final float[] getColumnTwoArray() {
        return colTwoArray;
    }
    
    public final int[] getColumnThreeArray() {
        return colThreeArray;
    }
    
    public final String[] getColumnThreeArrayStr() {
        return colThreeArrayStr;
    }
    
    public final float getColumn12Ratio() {
        return col12Ratio;
    }
    
    public final float[] getPlainRatioArray(){
        return plainRatioArray;
    }
    
    public final float[] getLogProdArray(){
        return logProdArray;
    }
    
    public final float[] getLogRatioArray(){
        return logRatioArray;
    }
    
    public final float[] getFlipLogRatioArray(){
        for (int i = 0; i < logRatioArray.length; i++){
            logRatioArray[i] *= -1;
        }
        return logRatioArray;
    }
    
    public final boolean[] getRowFlagArray() {
        return rowFlag;
    }
    
    public final int getRowCounts() {
        return count;
    }
    
    public final int getNonZeroCounts() {
        return nonZeroCount;
    }
    
    public final boolean[] getChannelFlagOne(){
        return channelFlag1;
    }
    
    public final boolean[] getChannelFlagTwo(){
        return channelFlag2;
    }
    
    public final String[] getUID(){
        return uidArrayStr;
    }
    
    public final void setFileName(String f){
        fileName = f;
    }
    
    public final void setColOneArray(float[] a){
        colOneArray = a;
    }
    
    public final void setColTwoArray(float[] b){
        colTwoArray = b;
    }
    
    public final void setRowFlag(boolean[] c){
        rowFlag = c;
    }
    
    public final void setColumnOneArrayRaw(int size) {
        colOneArrayRaw = new float[size];
    }
    
    public final void setColumnTwoArrayRaw(int size) {
        colTwoArrayRaw = new float[size];
    }
    
    public final void setNdxArray(int[] d){
        ndxArray = d;
    }
    
    public final void setUIDArray(String[] uid){
        repNotCombined = false;
        uidArrayStr = uid;
    }
}
