/*
 *   MathTools  -- A collection of useful math utility routines.
 *
 *   Copyright (C) 1999,2000,2001 by Joseph A. Huwaldt <jhuwaldt@gte.net>.
 *   All rights reserved.
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 **/
package org.tigr.midas.math;

import java.util.BitSet;


/**
 *  <p>  A collection of useful static routines of a general
 *       mathematical nature.  This file includes functions that
 *       accomplish all types of wonderous mathematical stuff.
 *  </p>
 *
 *  <p>  Modified by:  Joseph A. Huwaldt  </p>
 *
 *  @author   Joseph A. Huwaldt   Date:  September 29, 1997
 *  @version  February 14, 2001
 **/
public class MathTools extends Object {
    
    /**
     *  The natural logarithm of 10.
     **/
    public static final double LN10 = Math.log(10);
    
    /**
     *  The natural logarithm of 2.
     **/
    public static final double LN2 = Math.log(2);
    
    
    //-----------------------------------------------------------------------------------
    /**
     *  Test to see if a given long integer is even.
     *
     *  @param   n  Integer number to be tested.
     *  @return  True if the number is even, false if it is odd.
     **/
    public static final boolean even( long n ) {
        return (n & 1) == 0;
    }
    
    /**
     *  Test to see if a given long integer is odd.
     *
     *  @param   n  Integer number to be tested.
     *  @return  True if the number is odd, false if it is even.
     **/
    public static final boolean odd( long n ) {
        return (n & 1) != 0;
    }
    
    /**
     *  Calculates the square (x^2) of the argument.
     *
     *  @param   x  Argument to be squared.
     *  @return  Returns the square (x^2) of the argument.
     **/
    public static final float sqr( float x ) {
        if ( x == 0.f)
            return (0.f);
        else
            return x * x;
    }
    
    /**
     *  Computes the cube root of the specified real number.
     *  If the argument is negative, then the cube root is negative.
     *
     *  @param   x  Argument for which the cube root is to be found.
     *  @return  The cube root of the argument is returned.
     **/
    public static final float cubeRoot( float x ) {
        float value = 0;
        
        if ( x < 0. )
            value = -(float)Math.exp( Math.log(-x) / 3. );
        else
            value = (float)Math.exp( Math.log( x ) / 3. );
        
        return value;
    }
    
    public static final float nthRoot(float x, long n) {
        float value = 0;
        
        if (even(n)){
            if ( x > 0.){
                value = (float)Math.exp(Math.log(x) / n);
            }else if (x == 0){
                value = 0.f;
            }else{
                System.out.print("Invalid number!");
                value = 0.f;
            }
        }else{
            if (x > 0.){
                value = (float)Math.exp(Math.log(x) / n);
            }else if (x == 0){
                value = 0.f;
            }else{
                value = -(float)Math.exp(Math.log(-x) / n);
            }
        }
        return value;
    }
    
    /**
     *  Returns a number "a" raised to the power "b".  A "long" version
     *  of Math.pow().  This is much faster than using Math.pow() if
     *  the operands are integers.
     *
     *  @param   a  Number to be raised to the power "b".
     *  @param   b  Power to raise number "a" to.
     *  @return  A long integer "a" raised to the integer power "b".
     *  @throws  ArithmeticException if "b" is negative.
     **/
    public static final long lpow( long a, long b ) throws ArithmeticException {
        if ( b < 0 )
            throw new ArithmeticException( "Exponent must be positive." );
        
        long r = 1;
        while ( b != 0 ) {
            if ( odd( b ) )
                r *= a;
            
            b >>>= 1;
            a *= a;
        }
        return r;
    }
    
    /**
     *  Raises 2 to the small integer power indicated (eg:  2^3 = 8).
     *  This is MUCH faster than calling Math.pow(2, x).
     *
     *  @param   x  Amount to raise 2 to the power of.
     *  @return  Returns 2 raised to the power indicated.
     **/
    public static final long pow2( long x ) {
        long value = 1;
        for ( long i = 0; i < x; ++i ) {
            value *= 2;
        }
        return value;
    }
    
    /**
     *  Raises 10 to the small integer power indicated (eg: 10^5 = 100000).
     *  This is faster than calling Math.pow(10, x).
     *
     *  @param   x  Amount to raise 10 to the power of.
     *  @return  Returns 10 raised to the power indicated.
     **/
    public static final float pow10(int x) {
        float pow10 = 10.f;
        
        if (x != 0) {
            boolean neg = false;
            if (x < 0) {
                x *= -1;
                neg = true;
            }
            
            for (int i=1; i < x; ++i)
                pow10 *= 10.f;
            
            if (neg)
                pow10 = (float)1./pow10;
            
        } else
            pow10 = 1.f;
        
        return(pow10);
    }
    
    /**
     *  Find the base 10 logarithm of the given float.
     *
     *  @param   x  Value to find the base 10 logarithm of.
     *  @return  The base 10 logarithm of x.
     **/
    public static final float log10( float x ) {
        return (float)(Math.log(x)/LN10);
    }
    
    /**
     *  Find the base 2 logarithm of the given float.
     *
     *  @param   x  Value to find the base 2 logarithm of.
     *  @return  The base 2 logarithm of x.
     **/
    public static final float log2( float x ) {
        return (float)(Math.log(x)/LN2);
    }
    
    /**
     *  Rounds a floating point number up to the desired decimal place.
     *  Example:  1346.4667 rounded up to the 2nd place = 1400.
     *
     *  @param  value  The value to be rounded up.
     *  @param  place  Number of decimal places to round value to.
     *                 A place of 1 rounds to 10's place, 2 to 100's
     *                 place, et cetera.
     **/
    
    //Wei Liang modified -- 4/30/2003
    public static final float roundUpToPlace(float value, int place)  {
        
        float pow10;
        float holdvalue;
        
        if (place > 0){
            //System.out.println("place = " + place);
            pow10 = MathTools.pow10(place);	// = 10 ^ place

            if (value >=0){
                holdvalue = value * pow10;
                value = (float)Math.ceil(holdvalue);	// Round number up to nearest integer
                //System.out.println("value = " + value + ", holdvalue = " + holdvalue);
                if ( (value - holdvalue) > 0.5 ){
                    value = (float)Math.floor(holdvalue);	// Round number up to nearest integer
                }
                value /= pow10;
            }else{
                holdvalue = (-1) * value * pow10;
                value = (float)Math.ceil(holdvalue);
                if ( (value - holdvalue) > 0.5 ){
                    value = (float)Math.floor(holdvalue);	// Round number up to nearest integer
                }
                value /= pow10;
                value *= (-1);
            }
        }else if (place < 0){
            //System.out.println("place = " + place);
            place *= -1;
            pow10 = MathTools.pow10(place);	// = 10 ^ place
            if (value >=0){
                holdvalue = value / pow10;
                value = (float)Math.ceil(holdvalue);	// Round number up to nearest integer
                //System.out.println("value = " + value + ", holdvalue = " + holdvalue);
                if ( (value - holdvalue) > 0.5 ){
                    value = (float)Math.floor(holdvalue);	// Round number up to nearest integer
                }
                value *= pow10;
            }else{
                holdvalue = (-1) * value / pow10;
                value = (float)Math.ceil(holdvalue);	// Round number up to nearest integer
                if ( (value - holdvalue) > 0.5 ){
                    value = (float)Math.floor(holdvalue);	// Round number up to nearest integer
                }
                value *= pow10;
                value *= (-1);
            }

        }else{ //place = 0
            //System.out.println("place = " + place);
            if (value >= 0){
                value = (float)Math.ceil(value);
            }else{
                value = (float)Math.floor(value);
            }
        }
        
                /*	If the value is zero, just pass the number back out		*/
        //if (value != 0.) {
        
        //float pow10 = MathTools.pow10(place);	//	= 10 ^ place
        //float holdvalue = value/pow10;
        
        //value = Math.ceil(holdvalue);			// Round number up to nearest integer
        //value *= pow10;
        
        return(value);
    }
    
    /**
     *  Calculates the greatest common divisor between two input
     *  integers.  The GCD is the largest number that can be
     *  divided into both input numbers.  Uses Euler's method.
     *
     *  @param   xval  First integer
     *  @param   yval  Second integer
     *  @return  The largest number that can be divided into both input
     *           values.
     **/
    public static final long greatestCommonDivisor( long xval, long yval ) {
        long value = 0;
        while ( value != xval ) {
            if ( xval < yval )
                yval = yval - xval;
            
            else {
                if ( xval > yval )
                    xval = xval - yval;
                else
                    value = xval;
            }
        }
        return (value);
    }
    
    /**
     *  Returns the fractional part of a floating point number
     *  (removes the integer part).
     *
     *  @param   x  Argument for which the fractional part is to be returned.
     *  @return  The fractional part of the argument is returned.
     **/
    public static final float frac( float x ) {
        x = x - (long) x;
        if ( x < 0. )
            ++x;
        
        return x;
    }
    
    /**
     *  Straight linear 1D interpolation between two points.
     *
     *  @param   x1,y1  Coordinates of the 1st point (the high point).
     *  @param   x2,y2  Coordinates of the 2nd point (the low point).
     *  @param   x      The X coordinate of the point for which we want to
     *                  interpolate to determine a Y coordinate.  Will
     *                  extrapolate if X is outside of the bounds of the
     *                  point arguments.
     *  @return  The interpolated Y value corresponding to the input X
     *           value is returned.
     **/
    public static final float lineInterp( float x1, float y1, float x2,
    float y2, float x ) {
        return ((y2 - y1) / (x2 - x1) * (x - x1) + y1);
    }
    
    /**
     *  Converts a positive decimal number to it's binary
     *  equivelant.
     *
     *  @param  decimal  The positive decimal number to be encoded in
     *                   binary.
     *  @param  bits     The bitset to encode the number in.
     **/
    public static final void dec2bin( int decimal, BitSet bits ) {
        if ( decimal < 0 )
            throw new IllegalArgumentException( "Cannot convert a negative number to binary." );
        
        int i = 0;
        int value = decimal;
        while ( value > 0 ) {
            if ( value % 2 > 0 )
                bits.set( i );
            
            else
                bits.clear( i );
            
            value /= 2;
            ++i;
        }
        
        for ( i = i; i < bits.size(); ++i ) {
            bits.clear( i );
        }
    }
    
    /**
     *  Converts binary number to it's base 10 decimal equivelant.
     *
     *  @param   bits  The bitset that encodes the number to be converted.
     *  @return  Returns the decimal equivelent of the given binary number.
     **/
    public static final long bin2dec( BitSet bits ) {
        long value = 0;
        int length = bits.size();
        
        for ( int i = 0; i < length; ++i ) {
            if ( bits.get( i ) )
                value += pow2( i );
        }
        return value;
    }
    
    public static float hypot(float a, float b) {
	double r;
	if (Math.abs(a) > Math.abs(b)) {
	    r = b/a;
	    r = (float)(Math.abs(a)*Math.sqrt(1+r*r));
	} else if (b != 0) {
	    r = a/b;
	    r = (float)(Math.abs(b)*Math.sqrt(1+r*r));
	} else {
	    r = 0.0f;
	}
	return(float)r;
    }
    
    /**
     *  Used to test out the methods in this class.
     **/
    public static void main(String args[]) {
        
        //System.out.println();
        //System.out.println("Testing MathTools...");
        
        //System.out.println("  2 is an " + (even(2) ? "even" : "odd") + " number.");
        //System.out.println("  3 is an " + (odd(3) ? "odd" : "even") + " number.");
        //System.out.println("  The square of 3.8 is " + sqr(3.8) + ".");
        //System.out.println("  The cube root of 125 is " + cubeRoot(125) + ".");
        //System.out.println("  The integer 3^7 is " + lpow(3,7) + ".");
        //System.out.println("  The integer 2^8 is " + pow2(8) + ".");
        //System.out.println("  The float 10^-3 is " + pow10(-3) + ".");
        //System.out.println("  The base 10 logarithm of 8 is " + log10(8) + ".");
        //System.out.println("  Pow10 of 1.23456 = " + pow10(1.23456));
        //System.out.println("  Pow(10, 1.23456) = " + Math.pow(10.0, 1.23456));
        //System.out.println("  The base 2 logarithm of 2.8095 is " + log2(2.8095) + ".");
        System.out.println("  1346.466745676 rounded up to the nearest -1 is " + roundUpToPlace(1346.466745676f, -1));
        System.out.println("  1346.466745676 rounded up to the nearest 1 is " + roundUpToPlace(1346.466745676f, 1));
        System.out.println("  1346.466745676 rounded up to the nearest 0 is " + roundUpToPlace(1346.466745676f, 0));
        System.out.println("  -1346.466745676 rounded up to the nearest -1 is " + roundUpToPlace(-1346.466745676f, -1));
        System.out.println("  -1346.466745676 rounded up to the nearest 1 is " + roundUpToPlace(-1346.466745676f, 1));
        System.out.println("  -1346.466745676 rounded up to the nearest 0 is " + roundUpToPlace(-1346.466745676f, 0));
        //System.out.println("  -1346.4667 ceil is " + new Float(Math.ceil(-1346.4667)).intValue());
        //System.out.println("  -1346.5667 rint is " + Math.rint(-1346.5667));
        //System.out.println("  1346.5667 rint is " + Math.rint(1346.5667));
        //System.out.println("  The GCD of 125 and 45 is " + greatestCommonDivisor(125,45) + ".");
        //System.out.println("  The fractional part of 3.141593 is " + frac(3.141593) + ".");
        //System.out.println("  The 12th root of 1234 is " + nthRoot(1234, 12));
        
        //System.out.println(" old1: " + log2(136./130.));
        //System.out.println(" new1: " + log2(400188./216449.));
        //System.out.println(" new2: " + log2(401342./216449.));
        //System.out.println(" new3: " + log2(404384./216449.));
        //System.out.println(" new1/old1 = " + log2(400188./216449.)/log2(402151./216449.));
        //System.out.println(" new2/new1 = " + log2(401342./216449.)/log2(400188./216449.));
        //System.out.println(" new2/old1 = " + log2(404384./216449.)/log2(402151./216449.));
    }
    
}


